<?php
require_once(plugin_dir_path(__DIR__) . "libraries/stripe/init.php");
class QLCostCalculator_StripePayment
{
	private $stripe_api_secret_key;
	
	public function __construct($stripe_api_secret_key)
	{
		$this->stripe_api_secret_key = $stripe_api_secret_key;
	}
	
	public function createCustomer($email, $name, $phone)
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		return $stripe->customers->create([
			'email' => $email,
			'name' => $name,
			'phone' => $phone
		]);
	}
	
	public function createCheckoutSession($customer, $payment_currency, $product_name, $unit_amount, $customer_email, $success_url, $cancel_url, $recurring_interval="one_time", $recurring_interval_count=1)
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		$recurring = [];
		if($recurring_interval!="one_time")
		{
			$recurring = [
				'interval' => $recurring_interval,
				'interval_count' => $recurring_interval_count
			];
		}
		try 
		{
			return $stripe->checkout->sessions->create([
				//'expires_at' => 1691152896, //unix timestamp max 24h from session start, default value 24h
				//'client_reference_id' => uniqid("", true),
				'customer' => $customer,
				'line_items' => [[
					'price_data' => [
						'currency' => $payment_currency,
						'product_data' => [
							'name' => $product_name
						],
						'recurring' => $recurring,
						'unit_amount' => $unit_amount,
					],
					'quantity' => 1,
				]],
				'mode' => ($recurring_interval!="one_time" ? 'subscription' : 'payment'),
				'success_url' => $success_url,
				'cancel_url' => $cancel_url
			]);
		}
		catch(\Stripe\Exception\ApiErrorException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
	}
	
	public function getCheckoutSession($session_id, $status = null)
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		try 
		{
			$result = array();
			$checkout_session = $stripe->checkout->sessions->retrieve($session_id);
			if(!empty($checkout_session))
			{
				if(!empty($checkout_session->customer))
				{
				  $customer = $stripe->customers->retrieve($checkout_session->customer);
				  $result["customer_name"] = $customer->name;
				  $result["customer_email"] = $customer->email;
				}
				else
				{
					$result["customer_name"] = $checkout_session->customer_details->name;
					$result["customer_email"] = $checkout_session->customer_details->email;
				}
				$result["id"] = $checkout_session->id;
				$result["payment_intent"] = $checkout_session->payment_intent;
				$result["invoice"] = $checkout_session->invoice;
				$result["subscription"] = $checkout_session->subscription;
				$result["amount_total"] = $checkout_session->amount_total;
				$result["currency"] = $checkout_session->currency;
				$result["url"] = $checkout_session->url;
				$result["payment_status"] = $checkout_session->payment_status;
				$result["mode"] = $checkout_session->mode;
				// print_r($checkout_session->payment_intent);
				/*$intent = $stripe->paymentIntents->retrieve(
				  $checkout_session->payment_intent,
				  []
				);
				print_r($intent);*/
				/*echo '<a href="' . $checkout_session->url . '" target="_blank">pay now</a>';
				http_response_code(200);*/
			}
			$result["checkout_session_object"] = $checkout_session;
			return $result;
		} 
		catch(Error $e)
		{
			http_response_code(500);
			echo json_encode(['error' => $e->getMessage()]);
		}
	}
	
	public function createWebhookEndpoint()
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		try
		{
			$webhook_endpoint = $stripe->webhookEndpoints->create([
				'url' => get_home_url() . "?action=payment_webhook",
				'enabled_events' => [
					'payment_intent.amount_capturable_updated',
					'payment_intent.canceled',
					'payment_intent.created',
					'payment_intent.partially_funded',
					'payment_intent.payment_failed',
					'payment_intent.processing',
					'payment_intent.requires_action',
					'payment_intent.succeeded',
					'customer.subscription.created',
					'customer.subscription.deleted',
					'customer.subscription.paused',
					'customer.subscription.resumed',
					'customer.subscription.updated'/*,
					'checkout.session.async_payment_failed',
					'checkout.session.async_payment_succeeded',
					'checkout.session.completed',
					'checkout.session.expired',
					'invoice.created',
					'invoice.paid',
					'invoice.payment_failed',
					'subscription_schedule.canceled',
					'subscription_schedule.completed',
					'subscription_schedule.created',
					'customer.created',
					'customer.subscription.created'*/
				]
			]);
			return $webhook_endpoint;
		}
		catch(Stripe\Exception\InvalidRequestException $e)
		{
			return sprintf(__("Stripe Webhook has not been created! Error message: %s", 'cost-calculator'), esc_html($e->getError()->message));
		}
	}
	
	public function receiveWebhookEvent($endpoint_secret)
	{
		\Stripe\Stripe::setApiKey($this->stripe_api_secret_key);
		$payload = @file_get_contents('php://input');
		$sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];
		$event = null;
		try
		{
			$event = \Stripe\Webhook::constructEvent($payload, $sig_header, $endpoint_secret);
		}
		catch(\UnexpectedValueException $e)
		{
			//Invalid payload
			error_log($e);
			http_response_code(400);
			exit();
		}
		catch(\Stripe\Exception\SignatureVerificationException $e)
		{
			//Invalid signature
			error_log($e);
			http_response_code(400);
			exit();
		}
		if($this->updatePaymentEvent($event))
		{
			http_response_code(200);
		}
		else
		{
			http_response_code(400);
		}
		exit();
	}
	
	private function updatePaymentEvent($event)
	{
		if(!empty($event->data->object->id))
		{
			$meta_query = array();
			$args = array(
				'post_type' => 'calculations',
				'numberposts' => 1,
				'post_status' => 'any',
				'meta_query' => array(
					array(
						'key' => 'customer_id',
						'value' => $event->data->object->customer,
						'compare' => '='
					)
				)
			);			
			$calculations = get_posts($args);
			//$query = new WP_QUERY($args);
			//error_log($query->request);
			if(!empty($calculations))
			{
				if(strpos($event->type, 'payment_intent')!==false)
				{
					$payment_events = get_post_meta($calculations[0]->ID, "payment_events", true);
					if(!is_array($payment_events))
					{
						$payment_events = array();
					}
					$payment_events[] = array(
						'id' => $event->id,
						'type' => $event->type,
						'amount' => $event->data->object->amount,
						'amount_received' => $event->data->object->amount_received,
						'currency' => $event->data->object->currency,
						'time' => $event->created,
						'json' => json_encode($event)
					);
					update_post_meta($calculations[0]->ID, "payment_events", $payment_events);
					if($event->type=="payment_intent.succeeded" || $event->type=="payment_intent.payment_failed" || $event->type=="payment_intent.canceled")
					{
						$this->sendPaymentEmail($event, $calculations[0]);
					}
				}
				else if(strpos($event->type, 'customer.subscription')!==false)
				{
					$subscription_events = get_post_meta($calculations[0]->ID, "subscription_events", true);
					if(!is_array($subscription_events))
					{
						$subscription_events = array();
					}
					$subscription_events[] = array(
						'id' => $event->id,
						'type' => $event->type,
						'status' => $event->data->object->status,
						'current_period_start' => $event->data->object->current_period_start,
						'current_period_end' => $event->data->object->current_period_end,
						'time' => $event->created,
						'json' => json_encode($event)
					);
					update_post_meta($calculations[0]->ID, "subscription_events", $subscription_events);
				}
				return true;
			}
			return false;
		}
		return false;
	}
	
	private function sendPaymentEmail($event, $calculation)
	{
		ob_start();
		$result = array();
		$result["isOk"] = true;
		$cost_calculator_options = new QLCostCalculator_Options();
		$cost_calculator_contact_form_options = $cost_calculator_options->getContactFormOptions();
		$cost_calculator_payments_options = $cost_calculator_options->getPaymentsOptions();
		$checkout_session_id = get_post_meta($calculation->ID, "checkout_session_id", true);
		if((int)$cost_calculator_payments_options["send_payment_email"] || (int)$cost_calculator_payments_options["send_payment_email_client"])
		{
			$customer_name = "";
			$customer_email = "";
			if(isset($event->data->object->charges->data[0]))
			{
				if(!empty($event->data->object->charges->data[0]->billing_details->name))
				{
					$customer_name = $event->data->object->charges->data[0]->billing_details->name;
				}
				if(!empty($event->data->object->charges->data[0]->billing_details->email))
				{
					$customer_email = $event->data->object->charges->data[0]->billing_details->email;
				}
			}
			else if(isset($event->data->object->customer_details))
			{
				if(!empty($event->data->object->customer_details->name))
				{
					$customer_name = $event->data->object->customer_details->name;
				}
				if(!empty($event->data->object->customer_details->email))
				{
					$customer_email = $event->data->object->customer_details->email;
				}
			}
			if(!empty($checkout_session_id))
			{
				$checkout_session = $this->getCheckoutSession($checkout_session_id);
			}
			if(!empty($checkout_session) && ($customer_name=="" || $customer_email==""))
			{
				if($customer_name=="")
				{
					$customer_name = $checkout_session["customer_name"];
				}
				if($customer_email=="")
				{
					$customer_email = $checkout_session["customer_email"];
				}
			}
			$subscription_cancel_url = "";
			if(!empty($checkout_session["subscription"]))
			{
				$subscription_cancel_url = ((int)$cost_calculator_payments_options["subscription_cancel_page"]>0 ? get_permalink((int)$cost_calculator_payments_options["subscription_cancel_page"]) . "?type=page&" : get_home_url() . "?") . "action=cancel_subscription&stripe_subscription_id=" . esc_attr($checkout_session["subscription"]);
			}
		}
		//send to admin
		if((int)$cost_calculator_payments_options["send_payment_email"])
		{
			$subject = (!empty($cost_calculator_contact_form_options["payment_email_subject"]) ? $cost_calculator_contact_form_options["payment_email_subject"] : __("Payment of [amount] [currency] [status] ([summary_label])", 'cost-calculator'));
			$subject = str_replace("[id]", esc_html($event->data->object->id), $subject);
			$subject = str_replace("[amount]", number_format_i18n($event->data->object->amount/100, 2), $subject);
			$subject = str_replace("[amount_received]", number_format_i18n($event->data->object->amount_received/100, 2), $subject);
			$subject = str_replace("[currency]", esc_html($event->data->object->currency), $subject);
			$subject = str_replace("[mode]", (!empty($checkout_session["mode"]) ? ($checkout_session["mode"]=="payment" ? esc_html__("One time payment", 'cost-calculator') : esc_html(ucfirst($checkout_session["mode"]))) : ""), $subject);
			$subject = str_replace("[subscription_id]", (!empty($checkout_session["subscription"]) ? esc_html($checkout_session["subscription"]) : ""), $subject);
			$subject = str_replace("[datetime]", date(get_option('date_format'). ' ' . get_option('time_format'), $event->time), $subject);
			$subject = str_replace("[status]", str_replace("payment_", "" , substr($event->type, strrpos($event->type, ".") + 1)), $subject);
			$subject = str_replace("[summary_label]", esc_html(get_post_meta($calculation->ID, "summary_label", true)), $subject);
			$subject = str_replace("[name]", esc_html($customer_name), $subject);
			$subject = str_replace("[email]", esc_html($customer_email), $subject);
			$subject = str_replace("[checkout_url]", (!empty($checkout_session["url"]) ? esc_url($checkout_session["url"]) : ""), $subject);
			$subject = str_replace("[subscription_cancel_url]", (!empty($subscription_cancel_url) ? esc_url($subscription_cancel_url) : ""), $subject);

			$body = $cost_calculator_contact_form_options["payment_email_template"];
			$body = str_replace("[id]", esc_html($event->data->object->id), $body);
			$body = str_replace("[amount]", number_format_i18n($event->data->object->amount/100, 2), $body);
			$body = str_replace("[amount_received]", number_format_i18n($event->data->object->amount_received/100, 2), $body);
			$body = str_replace("[currency]", esc_html($event->data->object->currency), $body);
			$body = str_replace("[mode]", (!empty($checkout_session["mode"]) ? ($checkout_session["mode"]=="payment" ? esc_html__("One time payment", 'cost-calculator') : esc_html(ucfirst($checkout_session["mode"]))) : ""), $body);
			$body = str_replace("[subscription_id]", (!empty($checkout_session["subscription"]) ? esc_html($checkout_session["subscription"]) : ""), $body);
			$body = str_replace("[datetime]", date(get_option('date_format'). ' ' . get_option('time_format'), $event->time), $body);
			$body = str_replace("[status]", str_replace("payment_", "" , substr($event->type, strrpos($event->type, ".") + 1)), $body);
			$body = str_replace("[summary_label]", esc_html(get_post_meta($calculation->ID, "summary_label", true)), $body);
			$body = str_replace("[name]", esc_html($customer_name), $body);
			$body = str_replace("[email]", esc_html($customer_email), $body);
			$body = str_replace("[checkout_url]", (!empty($checkout_session["url"]) ? esc_url($checkout_session["url"]) : ""), $body);
			$body = str_replace("[subscription_cancel_url]", (!empty($subscription_cancel_url) ? esc_url($subscription_cancel_url) : ""), $body);
			
			$headers[] = 'Reply-To: ' . $customer_name . ' <' . $customer_email . '>' . "\r\n";
			$headers[] = 'From: ' . (!empty($cost_calculator_contact_form_options["admin_name_from"]) ? $cost_calculator_contact_form_options["admin_name_from"] : $cost_calculator_contact_form_options["admin_name"]) . ' <' . (!empty($cost_calculator_contact_form_options["admin_email_from"]) ? $cost_calculator_contact_form_options["admin_email_from"] : $cost_calculator_contact_form_options["admin_email"]) . '>' . "\r\n";
			$headers[] = 'Content-type: text/html';
			if(wp_mail($cost_calculator_contact_form_options["admin_name"] . ' <' . $cost_calculator_contact_form_options["admin_email"] . '>', $subject, $body, $headers))
			{
				$result["submit_message"] = (!empty($cost_calculator_contact_form_options["thankyou_message"]) ? $cost_calculator_contact_form_options["thankyou_message"] : __("Thank you for contacting us", 'cost-calculator'));
			}
			else
			{
				$result["isOk"] = false;
				$result["error_message"] = $GLOBALS['phpmailer']->ErrorInfo;
				$result["submit_message"] = (!empty($cost_calculator_contact_form_options["error_message"]) ? $cost_calculator_contact_form_options["error_message"] : __("Sorry, we can't send this message", 'cost-calculator'));
			}
		}
		//send to client
		if((int)$cost_calculator_payments_options["send_payment_email_client"] && $customer_email!="")
		{
			$subject = (!empty($cost_calculator_contact_form_options["payment_email_client_subject"]) ? $cost_calculator_contact_form_options["payment_email_client_subject"] : __("Payment of [amount] [currency] [status] ([summary_label])", 'cost-calculator'));
			$subject = str_replace("[id]", esc_html($event->data->object->id), $subject);
			$subject = str_replace("[amount]", number_format_i18n($event->data->object->amount/100, 2), $subject);
			$subject = str_replace("[amount_received]", number_format_i18n($event->data->object->amount_received/100, 2), $subject);
			$subject = str_replace("[currency]", esc_html($event->data->object->currency), $subject);
			$subject = str_replace("[mode]", (!empty($checkout_session["mode"]) ? ($checkout_session["mode"]=="payment" ? esc_html__("One time payment", 'cost-calculator') : esc_html(ucfirst($checkout_session["mode"]))) : ""), $subject);
			$subject = str_replace("[subscription_id]", (!empty($checkout_session["subscription"]) ? esc_html($checkout_session["subscription"]) : ""), $subject);
			$subject = str_replace("[datetime]", date(get_option('date_format'). ' ' . get_option('time_format'), $event->time), $subject);
			$subject = str_replace("[status]", str_replace("payment_", "" , substr($event->type, strrpos($event->type, ".") + 1)), $subject);
			$subject = str_replace("[summary_label]", esc_html(get_post_meta($calculation->ID, "summary_label", true)), $subject);
			$subject = str_replace("[name]", esc_html($customer_name), $subject);
			$subject = str_replace("[email]", esc_html($customer_email), $subject);
			$subject = str_replace("[checkout_url]", (!empty($checkout_session["url"]) ? esc_url($checkout_session["url"]) : ""), $subject);
			$subject = str_replace("[subscription_cancel_url]", (!empty($subscription_cancel_url) ? esc_url($subscription_cancel_url) : ""), $subject);

			$body = $cost_calculator_contact_form_options["payment_email_client_template"];
			$body = str_replace("[id]", esc_html($event->data->object->id), $body);
			$body = str_replace("[amount]", number_format_i18n($event->data->object->amount/100, 2), $body);
			$body = str_replace("[amount_received]", number_format_i18n($event->data->object->amount_received/100, 2), $body);
			$body = str_replace("[currency]", esc_html($event->data->object->currency), $body);
			$body = str_replace("[mode]", (!empty($checkout_session["mode"]) ? ($checkout_session["mode"]=="payment" ? esc_html__("One time payment", 'cost-calculator') : esc_html(ucfirst($checkout_session["mode"]))) : ""), $body);
			$body = str_replace("[subscription_id]", (!empty($checkout_session["subscription"]) ? esc_html($checkout_session["subscription"]) : ""), $body);
			$body = str_replace("[datetime]", date(get_option('date_format'). ' ' . get_option('time_format'), $event->time), $body);
			$body = str_replace("[status]", str_replace("payment_", "" , substr($event->type, strrpos($event->type, ".") + 1)), $body);
			$body = str_replace("[summary_label]", esc_html(get_post_meta($calculation->ID, "summary_label", true)), $body);
			$body = str_replace("[name]", esc_html($customer_name), $body);
			$body = str_replace("[email]", esc_html($customer_email), $body);
			$body = str_replace("[checkout_url]", (!empty($checkout_session["url"]) ? esc_url($checkout_session["url"]) : ""), $body);
			$body = str_replace("[subscription_cancel_url]", (!empty($subscription_cancel_url) ? esc_url($subscription_cancel_url) : ""), $body);
			
			$headers_client[] = 'Reply-To: ' . $cost_calculator_contact_form_options["admin_name"] . ' <' . $cost_calculator_contact_form_options["admin_email"] . '>' . "\r\n";
			$headers_client[] = 'From: ' . (!empty($cost_calculator_contact_form_options["admin_name_from"]) ? $cost_calculator_contact_form_options["admin_name_from"] : $cost_calculator_contact_form_options["admin_name"]) . ' <' . (!empty($cost_calculator_contact_form_options["admin_email_from"]) ? $cost_calculator_contact_form_options["admin_email_from"] : $cost_calculator_contact_form_options["admin_email"]) . '>' . "\r\n";
			$headers_client[] = 'Content-type: text/html';
			$send_email_client_status = wp_mail($customer_name . ' <' . $customer_email . '>', $subject, $body, $headers_client);
			if(!(int)$cost_calculator_payments_options["send_payment_email"] && $send_email_client_status)
			{
				$result["submit_message"] = (!empty($cost_calculator_contact_form_options["thankyou_message"]) ? $cost_calculator_contact_form_options["thankyou_message"] : __("Thank you for contacting us", 'cost-calculator'));
			}
			else if(!(int)$cost_calculator_payments_options["send_payment_email"])
			{
				$result["isOk"] = false;
				$result["error_message"] = $GLOBALS['phpmailer']->ErrorInfo;
				$result["submit_message"] = (!empty($cost_calculator_contact_form_options["error_message"]) ? $cost_calculator_contact_form_options["error_message"] : __("Sorry, we can't send this message", 'cost-calculator'));
			}
		}
		$system_message = ob_get_clean();
		$result["system_message"] = $system_message;
		return @json_encode($result);
	}
	
	public function getPaymentIntent($payment_intent)
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		try 
		{
			return $stripe->paymentIntents->retrieve($payment_intent, []);
		}
		catch(\Stripe\Exception\ApiErrorException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
		catch(\Stripe\Exception\InvalidArgumentException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
	}
	
	public function getSubscription($subscription)
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		try 
		{
			return $stripe->subscriptions->retrieve($subscription, []);
		}
		catch(\Stripe\Exception\ApiErrorException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
		catch(\Stripe\Exception\InvalidArgumentException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
	}
	
	public function cancelSubscription($subscription)
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		try 
		{
			return $stripe->subscriptions->cancel($subscription, []);
		}
		catch(\Stripe\Exception\ApiErrorException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
		catch(\Stripe\Exception\InvalidArgumentException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
	}
	
	public function getProduct($price)
	{
		$stripe = new \Stripe\StripeClient($this->stripe_api_secret_key);
		try 
		{
			return $stripe->products->retrieve($price, []);
		}
		catch(\Stripe\Exception\ApiErrorException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
		catch(\Stripe\Exception\InvalidArgumentException $e)
		{
			return json_encode(['error' => $e->getMessage()]);
		}
	}
}